﻿/*
 * SD320_Command.h
 *
 *  Created on: Oct 29, 2020
 *      Author: Litvin
 * 
 * 2020_10_29 v1.0.0.0 - Первоначальное создание класса
 * 2020_11_20 v1.0.1.0 - Реализована команда GetFirmwareVersion
 *					   - Реализована команда JumpTo
 * 2020_11_26 v1.0.1.1 - Реализован вывод отладочной информации в серианый порт на скорости 115200
 * 2020_12_02 v1.0.2.0 - Добавлена команда управления мощностью лазера
 * 2020_12_04 v1.0.3.0 - Добавлена команда _3DPrinter_SetFilamentLengthMMToRealease
 * 					   - Добавлена команда _3DPrinter_SetLinearSpeedMmPerSec
 * 2020_12_04 v1.0.4.0 - Добавлена команда _3DPrinter_SetExtruderAndTableTemperatureCelsius(float extruderTemperature, float tableTemperature);
 *					   - Добавлена команда _3DPrinter_GetExtruderAndTableTemperatureCelsius_Request(void);
 *					   - Добавлена команда _3DPrinter_GetExtruderAndTableTemperatureCelsius_Response(float extruderTemperature, float tableTemperature);
 *					   - Добавлена команда _3DPrinter_SetAbsolutePositionFilamentStepper(float absoluteMm);
 *					   - Добавлена переменная ExtruderTemperature;
 *					   - Добавлена переменная TableTemperature;
 * 2020_12_08 v1.0.4.1 - Исправлена ошибка парсинга команды в функции HandlePacket()
 * 2020_12_14 v1.0.5.0 - Добавлена команда установки выносов инструмента Tool_SetTool
 */

#ifndef INC_SD320_COMMAND_H_
#define INC_SD320_COMMAND_H_

#ifdef ARDUINO
	#include <Arduino.h>
	#ifdef DEBUG
		#define DEBUGPRINT(x) Serial.print(x)
		#define DEBUGPRINTLN(x) Serial.println(x)
		#define DEBUGPRINT_WITHTEXT(text, x) Serial.print(text); Serial.print(x)
		#define DEBUGPRINTLN_WITHTEXT(text, x) Serial.print(text); Serial.println(x)
	#endif
#else
	#include "main.h"
#endif
#include "SimpleCRC.h"

// Проверка последнего кода последней сформированной команды. Если совпадают - без перерасчета возвращает указатель на массив, в котором эта команда сформирована
#define INC_SD320_COMMAND_CheckLastFormedCommandID(x) if(LastFormedCommandID == x) return Buffer

/// <summary>
/// Commands code
/// </summary>

#define SD320_Command_SetMaxSpeed 0x0002
#define SD320_Command_SetAcceleration 0x0003

#define SD320_Command_Move 0x0010
#define SD320_Command_MoveTo 0x0011
#define SD320_Command_JumpTo 0x0012
#define SD320_Command_CurrentPosition_Request 0x0013
#define SD320_Command_CurrentPosition_Response 0x0014

#define SD320_Command_jMove 0x0020
#define SD320_Command_jMoveTo 0x0021
#define SD320_Command_jCurrentPosition_Request 0x0022
#define SD320_Command_jCurrentPosition_Response 0x0023

#define SD320_Command_Tool_SetVacuumeCup 0x0030
#define SD320_Command_Tool_SetClaws 0x0031
#define SD320_Command_Tool_SetLaserPWM 0x0032
#define SD320_Command_Tool_Rotate 0x0033
#define SD320_Command_Tool_RotateTo 0x0034
#define SD320_Command_Tool_SetTool 0x0035

#define SD320_Command_RobotCalibrate 0x0001

#define SD320_Command_GetFirmwareVersion_Request 0x00C0
#define SD320_Command_GetFirmwareVersion_Response 0x00C1
#define SD320_Command_CommandDone 0x00C8

/*
 * Дополнительные команды для работы в режиме 3D принтера
 */
#define SD320_Command_3DPrinter_SetFilamentLengthMMToRealease 0x0D01
#define SD320_Command_3DPrinter_SetLinearSpeedMmPerSec 0x0D02
#define SD320_Command_3DPrinter_SetExtruderAndTableTemperatureCelsius 0x0D03
#define SD320_Command_3DPrinter_GetExtruderAndTableTemperatureCelsius_Request 0x0D04
#define SD320_Command_3DPrinter_GetExtruderAndTableTemperatureCelsius_Response 0x0D05
#define SD320_Command_3DPrinter_SetAbsolutePositionFilamentStepper 0x0D06
/*
 * Дополнительные команды для работы в режиме 3D принтера
 */


typedef uint8_t (*FourUint16ArgsFunctionPointer)(uint16_t, uint16_t, uint16_t, uint16_t);
typedef uint8_t (*TwoFloatArgsFunctionPointer)(float, float);
typedef uint8_t (*ThreeFloatArgsFunctionPointer)(float, float, float);
typedef uint8_t (*OneFloatArgsFunctionPointer)(float);
typedef uint8_t (*OneBoolArgsFunctionPointer)(bool);
typedef uint8_t (*OneUint8tArgsFunctionPointer)(uint8_t);
typedef uint8_t (*VoidArgsFunctionPointer)(void);

class SD320_Command
{
public:
/// <summary>
/// Buffer storing generated packet
/// </summary>
	static uint8_t Buffer[253];
/// <summary>
/// Размер в байтах последней сформированной команды
/// </summary>
	static uint8_t PacketSize;
/// <summary>
/// Переменные хранения текущего положения в углах или координатах
/// </summary>
	static float X_J1;
	static float Y_J2;
	static float Z_J3;

	static float ExtruderTemperature;
	static float TableTemperature;

	static uint16_t FirmWareVersion_FullRewriteVersion;
	static uint16_t FirmWareVersion_ChangeArchitectureVersion;
	static uint16_t FirmWareVersion_NewFunctionalityVersion;
	static uint16_t FirmWareVersion_BugFixingVersion;


/// <summary>
/// Bool flag that the last command done
/// </summary>
	static bool CommandDoneFlag;

	static ThreeFloatArgsFunctionPointer SetMaxSpeed_Handler;
	static ThreeFloatArgsFunctionPointer SetAcceleration_Handler;

/// <summary>
/// Handler function for Move command
/// 1st arg - x
/// 2nd arg - y
/// 3rd ard - z
/// </summary>
/// <returns>Returns 1 if handling complete correctly, otherwise - 0.</returns>
	static ThreeFloatArgsFunctionPointer Move_Handler;
/// <summary>
/// Handler function for MoveTo command
/// 1st arg - x
/// 2nd arg - y
/// 3rd ard - z
/// </summary>
/// <returns>Returns 1 if handling complete correctly, otherwise - 0.</returns>
	static ThreeFloatArgsFunctionPointer MoveTo_Handler;
	static ThreeFloatArgsFunctionPointer JumpTo_Handler;
	static VoidArgsFunctionPointer CurrentPosition_Request_Handler;
	static ThreeFloatArgsFunctionPointer CurrentPosition_Response_Handler;

/// <summary>
/// Handler function for jMove command
/// 1st arg - j1
/// 2nd arg - j2
/// 3rd ard - j3
/// </summary>
/// <returns>Returns 1 if handling complete correctly, otherwise - 0.</returns>
	static ThreeFloatArgsFunctionPointer jMove_Handler;
/// <summary>
/// Handler function for jMoveTo command
/// 1st arg - j1
/// 2nd arg - j2
/// 3rd ard - j3
/// </summary>
/// <returns>Returns 1 if handling complete correctly, otherwise - 0.</returns>
	static ThreeFloatArgsFunctionPointer jMoveTo_Handler;
	static VoidArgsFunctionPointer jCurrentPosition_Request_Handler;
	static ThreeFloatArgsFunctionPointer jCurrentPosition_Response_Handler;


	static OneBoolArgsFunctionPointer Tool_SetVacuumeCup_Handler;
	static OneBoolArgsFunctionPointer Tool_SetClaws_Handler;
/// <summary>
/// Handler function for ToolRotate command
/// 1st arg - j1
/// </summary>
/// <returns>Returns 1 if handling complete correctly, otherwise - 0.</returns>
	static OneFloatArgsFunctionPointer Tool_Rotate_Handler;
/// <summary>
/// Handler function for ToolRotateTo command
/// 1st arg - j1
/// </summary>
/// <returns>Returns 1 if handling complete correctly, otherwise - 0.</returns>
	static OneFloatArgsFunctionPointer Tool_RotateTo_Handler;

	static OneUint8tArgsFunctionPointer Tool_SetLaserPWM_Handler;

	static ThreeFloatArgsFunctionPointer Tool_SetTool_Handler;

/// <summary>
/// Handler function for RobotCalibrate command
/// </summary>
/// <returns>Returns 1 if handling complete correctly, otherwise - 0.</returns>
	static VoidArgsFunctionPointer RobotCalibrate_Handler;


	static VoidArgsFunctionPointer GetFirmwareVersion_Request_Handler;
	static FourUint16ArgsFunctionPointer GetFirmwareVersion_Response_Handler;
/// <summary>
/// Handler function for CommandDone command
/// </summary>
/// <returns>Returns 1 if handling complete correctly, otherwise - 0.</returns>
	static OneBoolArgsFunctionPointer CommandDone_Handler;


	/*
	 * Дополнительные команды для работы в режиме 3D принтера
	 */
	static OneFloatArgsFunctionPointer _3DPrinter_SetFilamentLengthMMToRealease_Handler;
	static OneFloatArgsFunctionPointer _3DPrinter_SetLinearSpeedMmPerSec_Handler;

	static TwoFloatArgsFunctionPointer _3DPrinter_SetExtruderAndTableTemperatureCelsius_Handler;
	static VoidArgsFunctionPointer _3DPrinter_GetExtruderAndTableTemperatureCelsius_Request_Handler;
	static TwoFloatArgsFunctionPointer _3DPrinter_GetExtruderAndTableTemperatureCelsius_Response_Handler;
	static OneFloatArgsFunctionPointer _3DPrinter_SetAbsolutePositionFilamentStepper_Handler;
	/*
	 * Дополнительные команды для работы в режиме 3D принтера
	 */














/// <summary>
/// Class initialization for master SPI
/// </summary>
	static void InitForMaster();

/// <summary>
/// Handle received packet
/// </summary>
/// <param name="buf">Received byte packet</param>
/// <returns>Returns 1 if handling correctly completed, otherwise - 0</returns>
	static uint8_t HandlePacket(uint8_t* buf);

	static uint8_t* SetMaxSpeed(float axis1Speed, float axis2Speed, float axis3Speed);

	static uint8_t* SetAcceleration(float axis1Acceleration, float axis2Acceleration, float axis3Acceleration);


/// <summary>
/// Form the packet for relative move by axis
/// </summary>
/// <param name="relative_X">Relative value of x axis</param>
/// <param name="relative_Y">Relative value of y axis</param>
/// <param name="relative_Z">Relative value of z axis</param>
/// <returns>Return packet size</returns>
	static uint8_t* Move(float relative_X, float relative_Y, float relative_Z);

/// <summary>
/// Form the packet for absolute move by axis
/// </summary>
/// <param name="absoluteX">Relative value of x axis</param>
/// <param name="absoluteY">Relative value of y axis</param>
/// <param name="absoluteZ">Relative value of z axis</param>
/// <returns>Return packet size</returns>
	static uint8_t* MoveTo(float absoluteX, float absoluteY, float absoluteZ);

	static uint8_t* JumpTo(float absoluteX, float absoluteY, float deltaZ);

/// <summary>
/// Формирует запрос на получение текущего положения
/// </summary>
/// <returns>Возвращает указатель на массив со сформированным пакетом</returns>
	static uint8_t* CurrentPosition_Request(void);

	static uint8_t* CurrentPosition_Response(float absoluteX, float absoluteY, float absoluteZ);



	static uint8_t* jMove(float relative_j1, float relative_j2, float relative_j3);

/// <summary>
/// Form the packet for relative move by joints
/// </summary>
/// <param name="x">Relative value of j1</param>
/// <param name="y">Relative value of j2</param>
/// <param name="z">Relative value of j3</param>
/// <returns>Return packet size</returns>
	static uint8_t* jMoveTo(float absolute_j1, float absolute_j2, float absolute_j3);

	static uint8_t* jCurrentPosition_Request(void);

	static uint8_t* jCurrentPosition_Response(float absoluteJ1, float absoluteJ2, float absoluteJ3);


/// <summary>
/// Form the packet for tool rotating
/// </summary>
/// <param name="angle">Angle for tool</param>
/// <returns>Return packet size</returns>
	static uint8_t* Tool_RotateTo(float absolute);
	static uint8_t* Tool_Rotate(float relative);
	static uint8_t* Tool_SetVacuumeCup(bool isEnable);
	static uint8_t* Tool_SetClaws(bool isEnable);
	static uint8_t* Tool_SetLaserPWM(uint8_t pwm);
	static uint8_t* Tool_SetTool(float TX0, float TY0, float TZ0);


/// <summary>
/// Form the packet for robot calibration
/// </summary>
/// <returns>Return packet size</returns>
	static uint8_t* RobotCalibrate(void);

	static uint8_t* GetFirmwareVersion_Request(void);
	static uint8_t* GetFirmwareVersion_Response(uint16_t FullRewriteVersion, uint16_t ChangeArchitectureVersion, uint16_t NewFunctionalityVersion, uint16_t BugFixingVersion);
/// <summary>
/// Form the packet for command done response (for slave)
/// </summary>
/// <returns>Return packet size</returns>
	static uint8_t* CommandDone(bool isDone);

	/*
	 * Дополнительные команды для работы в режиме 3D принтера
	 */
	static uint8_t* _3DPrinter_SetFilamentLengthMMToRealease(float LengthMm);
	static uint8_t* _3DPrinter_SetLinearSpeedMmPerSec(float SpeedMmPerSec);

	static uint8_t* _3DPrinter_SetExtruderAndTableTemperatureCelsius(float extruderTemperature, float tableTemperature);
	static uint8_t* _3DPrinter_GetExtruderAndTableTemperatureCelsius_Request(void);
	static uint8_t* _3DPrinter_GetExtruderAndTableTemperatureCelsius_Response(float extruderTemperature, float tableTemperature);
	static uint8_t* _3DPrinter_SetAbsolutePositionFilamentStepper(float absoluteMm);
	/*
	 * Дополнительные команды для работы в режиме 3D принтера
	 */


private:
/// <summary>
/// Код последней сформированной команды
/// </summary>
	static uint16_t LastFormedCommandID;

/// <summary>
/// Clear main buffer.
/// </summary>
	static void ClearBuffer();

/// <summary>
/// Sets in buffer command ID
/// </summary>
/// <param name="command">command ID</param>
	static void SetCommandID(uint16_t command);

/// <summary>
/// Added to buffer value from the value pointer of the current size
/// </summary>
/// <param name="value">Pointer to value</param>
/// <param name="valueByteSize">Value byte count</param>
	static void AddValue(uint8_t* value, uint8_t valueByteSize);

/// <summary>
/// Set ending length of the packet
/// </summary>
	static void SetLength();

/// <summary>
/// Calculate and added to buffer crc checksum
/// </summary>
	static void CalculateCRC();

};

#endif /* INC_SD320_COMMAND_H_ */
